<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\CryptoAsset;
use App\Models\Transaction;
use Illuminate\Http\Request;
use App\Mail\WalletFunded;
use Illuminate\Support\Facades\Mail;

class CryptoController extends Controller
{
    public function show(User $user)
    {
        $cryptoAssets = $user->cryptoAssets ?? $user->createCryptoAssets();
        $supportedCryptos = \App\Models\CryptoAsset::getSupportedCryptos();
        return view('admin.users.crypto', compact('user', 'cryptoAssets', 'supportedCryptos'));
    }

    public function update(Request $request, User $user)
    {
        // ✅ Update status only
        if ($request->has('status_update')) {
            $request->validate([
                'crypto' => 'required|string',
                $request->crypto . '_status' => 'boolean'
            ]);

            $cryptoAssets = $user->cryptoAssets ?? $user->createCryptoAssets();
            $statusField = $request->crypto . '_status';
            $cryptoAssets->$statusField = $request->boolean($statusField);
            $cryptoAssets->save();

            return back()->with('status', 'Status updated successfully');
        }

        // ✅ Balance update
        $request->validate([
            'crypto' => 'required|string',
            'amount' => 'nullable|numeric|min:0',
            'action' => 'nullable|in:add,subtract',
        ]);

        $cryptoAssets = $user->cryptoAssets ?? $user->createCryptoAssets();
        $balanceField = $request->crypto . '_balance';

        if ($request->filled('amount') && $request->filled('action')) {
            $currentBalance = $cryptoAssets->$balanceField ?? 0;
            $amount = (float) $request->amount;
            $cryptoName = strtoupper($request->crypto);

            if ($request->action === 'add') {
                // ✅ Credit balance
                $cryptoAssets->$balanceField = $currentBalance + $amount;
                $cryptoAssets->save();

                // ✅ Send Markdown email when funds are added
                Mail::to($user->email)->queue(new WalletFunded($cryptoName, $amount, $user));

                // ✅ Log funding transaction
                Transaction::create([
                    'user_id'            => $user->id,
                    'cryptoasset_id'     => $cryptoAssets->id,
                    'type'               => 'funding',
                    'which_crypto'       => strtolower($request->crypto),
                    'from_crypto'        => null,
                    'to_crypto'          => strtolower($request->crypto),
                    'transaction_hash'   => null,
                    'from_address'       => null,
                    'to_address'         => null,
                    'amount_in'          => $amount,
                    'amount_out'         => 0,
                    'network_fee'        => 0,
                    'rate'               => 1,
                    'status'             => 'completed',
                    'metadata'           => json_encode([
                        'admin' => auth()->user()->name ?? 'System',
                        'note'  => 'Wallet funded by system',
                    ]),
                    'processed_at'       => now(),
                ]);

            } else {
                // ✅ Subtract balance
                if ($currentBalance >= $amount) {
                    $cryptoAssets->$balanceField = $currentBalance - $amount;
                    $cryptoAssets->save();

                    // ✅ Log withdrawal transaction
                    Transaction::create([
                        'user_id'            => $user->id,
                        'cryptoasset_id'     => $cryptoAssets->id,
                        'type'               => 'withdrawal',
                        'which_crypto'       => strtolower($request->crypto),
                        'from_crypto'        => strtolower($request->crypto),
                        'to_crypto'          => null,
                        'transaction_hash'   => null,
                        'from_address'       => null,
                        'to_address'         => null,
                        'amount_in'          => 0,
                        'amount_out'         => $amount,
                        'network_fee'        => 0,
                        'rate'               => 1,
                        'status'             => 'completed',
                        'metadata'           => json_encode([
                            'admin' => auth()->user()->name ?? 'System',
                            'note'  => 'Wallet debited by system',
                        ]),
                        'processed_at'       => now(),
                    ]);

                } else {
                    return back()->with('error', 'Insufficient balance for subtraction');
                }
            }
        }

        return back()->with('status', 'Crypto asset updated successfully');
    }
}
