<?php

namespace App\Livewire\Auth;

use Livewire\Component;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;

class PasscodeSetup extends Component
{
    public $passcode = '';
    public $confirmPasscode = '';
    public $step = 1; // 1 = Enter passcode, 2 = Confirm passcode
    public $error = '';
    public $tempPasscode = '';

    protected $rules = [
        'passcode' => ['required', 'digits:6'],
        'confirmPasscode' => ['required', 'digits:6'],
    ];

    public function mount()
    {
        $redirect = $this->checkPasscodeStatus();
        if ($redirect) return $redirect;
    }

    public function checkPasscodeStatus()
{
    $user = auth()->user();
    if ($user->hasPasscodeSetup() && !is_null($user->passcode)) {
        if (!session()->has('passcode_verified')) {
            return redirect()->route('passcode.show');
        }
        return redirect()->route('dashboard')
            ->with('status', 'Passcode is already set.');
    }
}


    public function addNumber($number)
    {
        if ($this->step === 1 && strlen($this->passcode) < 6) {
            $this->passcode .= $number;
            $this->error = '';

            if (strlen($this->passcode) === 6) {
                $this->validateOnly('passcode');
                $this->tempPasscode = $this->passcode;
                $this->passcode = '';
                $this->step = 2;
            }
        } elseif ($this->step === 2 && strlen($this->confirmPasscode) < 6) {
            $this->confirmPasscode .= $number;
            $this->error = '';

            if (strlen($this->confirmPasscode) === 6) {
                $this->verifyAndSave();
            }
        }
    }

    public function deleteNumber()
    {
        if ($this->step === 1) {
            $this->passcode = substr($this->passcode, 0, -1);
        } else {
            $this->confirmPasscode = substr($this->confirmPasscode, 0, -1);
        }
        $this->error = '';
    }

    public function clearPasscode()
    {
        if ($this->step === 1) {
            $this->passcode = '';
        } else {
            $this->confirmPasscode = '';
        }
        $this->error = '';
    }

    public function goBack()
    {
        if ($this->step === 2) {
            $this->step = 1;
            $this->passcode = $this->tempPasscode;
            $this->confirmPasscode = '';
            $this->error = '';
        }
    }

    public function verifyAndSave()
    {
        if ($this->step === 1) {
            // Handle Step 1: Create Passcode
            $this->validate([
                'passcode' => ['required', 'digits:6'],
            ]);

            $this->tempPasscode = $this->passcode;
            $this->passcode = '';
            $this->step = 2;
        } elseif ($this->step === 2) {
            // Handle Step 2: Confirm Passcode
            $this->validate([
                'confirmPasscode' => ['required', 'digits:6'],
            ]);

            if ($this->confirmPasscode !== $this->tempPasscode) {
                $this->error = 'Passcodes do not match. Please try again.';
                $this->confirmPasscode = '';
                $this->step = 1; // Optionally, you can reset to step 1 or keep at step 2
                return;
            }

            // Save the passcode securely
            auth()->user()->update([
                'passcode' => Hash::make($this->confirmPasscode),
            ]);

            session(['passcode_verified' => true]);

            return redirect()->route('dashboard')
                ->with('status', 'Passcode set successfully.');
        }
    }

    public function render()
    {
        $redirect = $this->checkPasscodeStatus();
        if ($redirect) return $redirect;

        return view('livewire.auth.passcode-setup')
            ->layout('layouts.guest');
    }
}
